;;; -*- auto-recompile: t -*-

;;; timidity-cfg.el --- editing TiMidity++ configuration files

;; Keywords: timidity

;; This file is not part of GNU Emacs.
;; 
;; timidity-cfg.el is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; timidity-cfg.el is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Author: Stphane Rollandin <hepta@zogotounga.net>
;;---------

;;; Commentary:
;;-------------

;;    this mode is an add-on to the KeyKit mode package
;;    it allows for fast and interactive editing of TiMidity 
;;    configuration files
;;
;;    first of all, customize it by going to the group 'timidity-cfg
;;
;;    then ensure that a midi file is present to be used at a test:
;;    note that you can create one such file from emacs by selecting
;;    the item "set the test melody" in the Timidity menu (when in 
;;    timidity-cfg mode), and entering a KeyKit phrase expression 
;;    (this will only work if the lowkey settings are correct in
;;    the 'keykit-util customization group)
;;
;;    when the buffer is in read-only mode (which you can toggle from
;;    the Timidity menu), the following keys have special actions:
;;
;;    <RETURN> plays the test melody with the patch at point
;;             (or the drum pattern if it is a drum patch at point)
;;    t        lets you define a new test melody (a KeyKit phrase)
;;    d        lets you define a new drum pattern (a KeyKit phrase)
;;    b, n     jump to next bank/drumset
;;    p        jump to previous bank/drumset
;;    a        set amplification
;;    +        rise amplification by 50%
;;    -        lower amplification by 33%
;;    *        rise amplification by 50% for the whole bank/drumset
;;    /        lower amplification by 33% for the whole bank/drumset
;;    o        comment out the current line (delete a mix line)
;;    i        uncomment the current line
;;    s        stores the patch at point in a KeyKit variable
;;    e        stores the patch and the test melody as a KeyKit variable
;;    m        import the patch and the test melody as two Compositor boxes
;;    r        import the patch by replacing an existing Compositor box
;;

;;; Installation:
;;---------------                   
;;
;; install the rest of the keykit-mode package,
;; then add the following to your .emacs:
;;
;;   (require 'timidity-cfg)
;;
;; if you want a given *.cfg file to be edited in timidity-cfg mode
;; by default, add the following in its first line:
;;
;;     #-*-timidity-cfg-*-


;; last modified March 19, 2002



(defgroup timidity-cfg nil
  "Major mode for editing TiMidity++ configuration files"
  :group 'keykit
  :prefix "timcfg-")

(defcustom timcfg-timidity-executable-linux "timidity"
  "TiMidity++ binary for linux"
  :type 'string
  :group 'timidity-cfg)

(defcustom timcfg-timidity-executable-win "c:/timidity/timpp2100q.exe"
  "TiMidity++ executable for windows"
  :type 'string
  :group 'timidity-cfg)

(defcustom timcfg-timidity-options " -EFchorus=0 -EFreverb=0 -A80 -s44100 -p128 -Od "
  "TiMidity command-line options"
  :type 'string
  :group 'timidity-cfg)

(defcustom timcfg-midi-test-file "c:/timidity/test.mid"
  "MIDI file to be played when testing a patch"
  :type 'string
  :group 'timidity-cfg)

(defcustom timcfg-midi-drumtest-file "c:/timidity/drumtest.mid"
  "MIDI file to be played when testing a drum"
  :type 'string
  :group 'timidity-cfg)

(defcustom timcfg-midi-mix-file "c:/timidity/mix.mid"
  "temporary MIDI file used to play a mix"
  :type 'string
  :group 'timidity-cfg)

(defcustom timcfg-drum-template "'ad40,a,a,a'"
  "KeyKit phrase used as rythm template when testing a drum"
  :type 'string
  :group 'timidity-cfg)

(defvar timcfg-previous-drum-template "")
(defvar timcfg-previous-note "")
(defvar timcfg-melody "''")
(defvar timcfg-last-bank "''")
(defvar timcfg-last-patch "''")

(defcustom timcfg-bad-word "BEURK"
  "Label used when commenting out a patch.
Warning: do not use characters having special meanings in a regexp"
  :type 'string
  :group 'timidity-cfg)


;;; --------------------------- Keymap & menus -----------------------------

(defvar timidity-cfg-mode-map (make-sparse-keymap)
  "Keymap used in KK mode.")

(define-key timidity-cfg-mode-map "8" 'timcfg-electric-8)
(define-key timidity-cfg-mode-map "2" 'timcfg-electric-2)
(define-key timidity-cfg-mode-map "9" 'timcfg-electric-9)
(define-key timidity-cfg-mode-map "3" 'timcfg-electric-3)
(define-key timidity-cfg-mode-map "x" 'timcfg-electric-x)
(define-key timidity-cfg-mode-map "e" 'timcfg-electric-e)
(define-key timidity-cfg-mode-map "m" 'timcfg-electric-m)
(define-key timidity-cfg-mode-map "r" 'timcfg-electric-r)
(define-key timidity-cfg-mode-map "s" 'timcfg-electric-s)
(define-key timidity-cfg-mode-map "a" 'timcfg-electric-a)
(define-key timidity-cfg-mode-map "+" 'timcfg-amp-plus)
(define-key timidity-cfg-mode-map "-" 'timcfg-amp-minus)
(define-key timidity-cfg-mode-map "*" 'timcfg-amp-plus-b)
(define-key timidity-cfg-mode-map "/" 'timcfg-amp-minus-b)
(define-key timidity-cfg-mode-map "b" 'timcfg-electric-b)
(define-key timidity-cfg-mode-map "p" 'timcfg-electric-p)
(define-key timidity-cfg-mode-map "n" 'timcfg-electric-n)
(define-key timidity-cfg-mode-map "\r" 'timcfg-electric-return)
(define-key timidity-cfg-mode-map "o" 'timcfg-electric-o)
(define-key timidity-cfg-mode-map "i" 'timcfg-electric-i)
(define-key timidity-cfg-mode-map "t" 'timcfg-electric-t)
(define-key timidity-cfg-mode-map "d" 'timcfg-electric-d)

(define-key timidity-cfg-mode-map [menu-bar] (make-sparse-keymap))

(define-key timidity-cfg-mode-map [menu-bar cfg]
  (cons "TiMidity" (make-sparse-keymap "CFG-mode")))

(define-key timidity-cfg-mode-map [menu-bar cfg toggle-read-only]
  '("Toggle read only" . toggle-read-only))
(define-key timidity-cfg-mode-map [menu-bar cfg sep02]
   '("--"))
(define-key timidity-cfg-mode-map [menu-bar cfg set-amplification]
  '("Set amplification" . timcfg-write-amp))
(define-key timidity-cfg-mode-map [menu-bar cfg sep01]
   '("--"))
(define-key timidity-cfg-mode-map [menu-bar cfg set-drum-template]
  '("Define the drum template" . timcfg-change-drum-template))
(define-key timidity-cfg-mode-map [menu-bar cfg listen-drum]
  '("Listen to drum at point" . timcfg-listen-drum-at-point))
(define-key timidity-cfg-mode-map [menu-bar cfg set-melody]
  '("Define the test melody" . timcfg-change-melody))
(define-key timidity-cfg-mode-map [menu-bar cfg listen-patch]
  '("Listen to patch at point" . timcfg-listen-patch-at-point))
(define-key timidity-cfg-mode-map [menu-bar cfg sep00]
   '("--"))
(define-key timidity-cfg-mode-map [menu-bar cfg set-amplification]
  '("Describe mode" . describe-mode))


;;; --------------------------- major mode definition ---------------------------

(defun timidity-cfg-mode ()
  "Major mode for editing TiMidity++ configuration files. Requires the prior installation of keykit-mode and its correct setting.

When the buffer is in read-only mode (which you can toggle from the Timidity menu, or from the mode line), the following keys have special actions:

    <RETURN> plays the test melody with the patch at point
             (or the drum pattern if it is a drum patch at point)
    t        lets you define a new test melody (a KeyKit phrase)
    d        lets you define a new drum pattern (a KeyKit phrase)
    b, n     jump to next bank/drumset
    p        jump to previous bank/drumset
    a        set amplification
    +        rise amplification by 50%
    -        lower amplification by 33%
    *        rise amplification by 50% for the whole bank/drumset
    /        lower amplification by 33% for the whole bank/drumset
    o        comment out the current line (delete a mix line)
    i        uncomment the current line
    s        stores the patch at point in a KeyKit variable
    e        stores the patch and the test melody as a KeyKit variable
    m        import the patch and the test melody as two Compositor boxes
    r        import the patch by replacing an existing Compositor box

The corresponding bindings are:

\\{timidity-cfg-mode-map}

Turning on timidity-cgf mode calls the value of the variable timidity-cgf-mode-hook
with no args, if that value is non-nil.
"
  (interactive)
  (kill-all-local-variables)
  (use-local-map timidity-cfg-mode-map)
  (setq major-mode 'timidity-cfg-mode)
  (setq mode-name "CFG")
  (make-local-variable 'comment-start)
  (setq comment-start "#")
  (make-local-variable 'comment-end)
  (setq comment-end "")
  (toggle-read-only t)
  (timcfg-fontify)
  (run-hooks 'timidity-cfg-mode-hook))



;;; --------------------------- navigation -----------------------------

(defun timcfg-patch-line-p ()
  (and (timcfg-current-patch)
       (timcfg-current-bank)))

(defun timcfg-mix-line-p ()
  (save-excursion
    (beginning-of-line)
    (if (looking-at "^#\\^\\(.+\\)\\^[ \t]")
	(match-string 1)
      nil)))

(defun timcfg-current-bank ()
  (save-excursion
    (if (timcfg-previous-bank)
	(progn
	  (re-search-forward "[0-9]+")
	  (match-string 0))
    nil)))

(defun timcfg-current-bank-type ()
  (save-excursion
    (if (timcfg-previous-bank)
	(progn
	  (re-search-forward "^[a-z]+")
	  (match-string 0))
      nil)))

(defun timcfg-next-bank ()
  (interactive)
  (re-search-forward "^\\(bank\\|drumset\\)[ \t]*\\([0-9]+\\)" nil t))

(defun timcfg-previous-bank ()
  (interactive)
  (re-search-backward "^\\(bank\\|drumset\\)[ \t]*\\([0-9]+\\)" nil t))

(defun timcfg-current-patch ()
  (save-excursion
    (beginning-of-line) 
    (if (looking-at "[0-9]+") 
	(match-string 0)
      nil)))

(defun timcfg-end-of-bank ()
  (save-excursion
    (end-of-line)
    (goto-char (if (timcfg-next-bank) (point) (point-max)))
    (forward-line -1)
    (end-of-line)
    (point)))

;;; --------------------------- keymap commands -----------------------------

(defun timcfg-electric-b ()
  "go to next bank"
  (interactive)
  (if buffer-read-only (timcfg-next-bank)
    (insert "b")))

(defun timcfg-electric-n ()
  "go to next bank"
  (interactive)
  (if buffer-read-only (timcfg-next-bank)
    (insert "n")))

(defun timcfg-electric-p ()
  "go to previous bank"
  (interactive)
  (if buffer-read-only (timcfg-previous-bank)
    (insert "p")))

(defun timcfg-electric-m ()
  "import boxes into the Compositor tool"
  (interactive)
  (if buffer-read-only (timcfg-import-boxes)
    (insert "m")))

(defun timcfg-electric-r ()
  "import patch at point into the Compositor tool as a box replacement"
  (interactive)
  (if buffer-read-only (timcfg-replace-box)
    (insert "r")))

(defun timcfg-electric-s ()
  "store patch at point in a keyKit variable"
  (interactive)
  (if buffer-read-only (timcfg-snarf-patch nil)
    (insert "s")))

(defun timcfg-electric-e ()
  "store patch at point and test melody as a keyKit variable"
  (interactive)
  (if buffer-read-only (timcfg-snarf-patch t)
    (insert "e")))

(defun timcfg-electric-x ()
  "add a new component to a mix"
  (interactive)
  (if buffer-read-only (timcfg-add-to-mix)
    (insert "x")))

(defun timcfg-electric-8 ()
  "add a new component to a mix"
  (interactive)
  (if buffer-read-only (timcfg-mix-move-up 1)
    (insert "8")))

(defun timcfg-electric-9 ()
  "add a new component to a mix"
  (interactive)
  (if buffer-read-only (timcfg-mix-move-up 10)
    (insert "9")))

(defun timcfg-electric-2 ()
  "add a new component to a mix"
  (interactive)
  (if buffer-read-only (timcfg-mix-move-down 1)
    (insert "2")))

(defun timcfg-electric-3 ()
  "add a new component to a mix"
  (interactive)
  (if buffer-read-only (timcfg-mix-move-down 10)
    (insert "3")))

(defun timcfg-electric-return ()
  "Plays the current patch if any, else jump to next patch"
  (interactive)
  (if buffer-read-only
      (if (timcfg-patch-line-p)
	  (if (string= (timcfg-current-bank-type) "drumset")
	      (timcfg-listen-drum-at-point)
	    (if (string= (timcfg-current-bank-type) "bank")
		(timcfg-listen-patch-at-point)))
	(if (timcfg-mix-line-p)
	    (timcfg-listen-mix-at-point)
	  (re-search-forward "^[0-9]+" nil t)))
    (insert "\n")))

(defun timcfg-electric-t()
  "Invoke KeyKit for defining a new test melody."
  (interactive)
  (if buffer-read-only (timcfg-change-melody)
    (insert "t")))

(defun timcfg-electric-d()
  "Queries for a new drum template."
  (interactive)
  (if buffer-read-only (timcfg-change-drum-template)
    (insert "d")))

(defun timcfg-electric-o ()
  "comment out the current patch if any, delete a mix line"
  (interactive)
  (if buffer-read-only
      (if (timcfg-patch-line-p)
	  (with-read-only-status
	    (beginning-of-line)
	    (insert "#" timcfg-bad-word "  "))
	(when (timcfg-mix-line-p)
	  (timcfg-kill-line)))
    (insert "o")))

(defun timcfg-electric-i ()
  "uncomment the current patch"
  (interactive)
  (if buffer-read-only
      (with-read-only-status
       (beginning-of-line)
       (if (looking-at (concat "\\(#" timcfg-bad-word "[ \t]*\\)[0-9]+"))
	   (replace-match "" t t nil 1)))
    (insert "i")))

(defun timcfg-amp-plus-b ()
  "rise amplification by 50% in whole bank"
  (interactive)
  (if buffer-read-only
      (with-whole-bank
       (timcfg-write-amp (round (* 1.5 (timcfg-read-amp)))))
    (insert "*")))

(defun timcfg-amp-minus-b ()
  "lower amplification by 33% in whole bank"
  (interactive)
  (if buffer-read-only
      (with-whole-bank
	(timcfg-write-amp (round (* 2 (/ (timcfg-read-amp) 3)))))
    (insert "/")))

(defun timcfg-amp-plus ()
  "rise amplification by 50%"
  (interactive)
  (if (and buffer-read-only
	   (timcfg-patch-line-p))
	(timcfg-write-amp (round (* 1.5 (timcfg-read-amp))))
    (insert "+")))

(defun timcfg-amp-minus ()
  "lower amplification by 33%"
  (interactive)
  (if (and buffer-read-only
	   (timcfg-patch-line-p))
	(timcfg-write-amp (round (* 2 (/ (timcfg-read-amp) 3))))
    (insert "-")))

(defun timcfg-read-amp ()
  (save-excursion
    (beginning-of-line)
    (if (re-search-forward "amp=\\([0-9]+\\)" (save-excursion (end-of-line) (point)) t)
	(string-to-number (match-string 1))
      (if (timcfg-patch-line-p)
	  100
	nil))))

(defun timcfg-write-amp (&optional val)
  (interactive)
  (let ((sval (if val (number-to-string val)
		(read-from-minibuffer "amp="))))
    (if (timcfg-patch-line-p)
      (with-read-only-status
       (beginning-of-line)
       (let ((beg (point)))
	 (if (re-search-forward "amp=\\([0-9]+\\)" 
				(save-excursion (end-of-line) (point)) t)    
	     (replace-match sval t t nil 1)
	   (end-of-line)
	   (search-backward "#" beg t)
	   (skip-chars-backward " \t")
	   (insert " amp=" sval)))))))

(defun timcfg-electric-a ()
  "set the current amplification"
  (interactive)
  (if buffer-read-only
      (timcfg-write-amp)
    (insert "a")))

(defmacro with-read-only-status (&rest body)
  `(let ((read-only-status buffer-read-only))
     (if read-only-status
	 (toggle-read-only))
     ,@body
     (if read-only-status
	 (toggle-read-only))))

    
(defmacro with-whole-bank (&rest body)
 `(save-excursion
    (timcfg-previous-bank)
    (while (< (point) (timcfg-end-of-bank))
      (forward-line 1)
      (if (timcfg-patch-line-p)
	  ,@body))))

;;; --------------------------- menu items -----------------------------

(defun timcfg-listen-drum-at-point ()
  "Call TiMidity and play the drum at point.
This only works in drumsets: for a regular patch, use timcfg-listen-patch-at-point"
  (interactive)
  (let ((midi-test timcfg-midi-drumtest-file)
	(note (save-excursion
		 (beginning-of-line) 
		 (word-at-point)))
 	(drumset (save-excursion
		   (if (re-search-backward "^drumset[ \t]*\\([0-9]+\\)" nil t)
		       (match-string 1)
		     nil))))
    (if drumset
	(progn
	  (unless (and (string= timcfg-drum-template timcfg-previous-drum-template)
		       (string= note timcfg-previous-note))
	    (with-temp-buffer
	      (insert "ph = " timcfg-drum-template "\n"
		      "ph.chan = 10 \n"
		      "ph.pitch = " note "\n"
		      "ph = progchange(" drumset " + 1,10) + ph \n"
		      "writemf(ph, \"" timcfg-midi-drumtest-file "\")")
	      (kk-run-lowkey "1" 1 (point-max))))
	  (setq timcfg-previous-drum-template timcfg-drum-template
		timcfg-previous-note note)
	  (shell-command
	   (concat (if (equal system-type 'gnu/linux)
		       timcfg-timidity-executable-linux
		     timcfg-timidity-executable-win)
		   " " timcfg-timidity-options " " midi-test " &")))
      (message "no drumset specified !"))))
  

(defun timcfg-change-drum-template (&optional melody)
  "Queries for a new drum template."
  (interactive)
  (setq timcfg-drum-template (or melody
				 (read-from-minibuffer "KeyKit phrase> "))))


(defun timcfg-listen-patch-at-point (&optional midi-test)
  "Call TiMidity and play a few notes with the patch at point.
This only works in banks: for a drumset, use timcfg-listen-drum-at-point"
  (interactive)
  (let ((midi-test (or midi-test timcfg-midi-test-file))
	(patch (save-excursion
		 (beginning-of-line) 
		 (word-at-point)))
	(bank (save-excursion
		(if (re-search-backward "^bank[ \t]*\\([0-9]+\\)" nil t)
		    (match-string 1)
		  nil))))
    (if bank
	(shell-command
	 (concat (if (equal system-type 'gnu/linux)
		     timcfg-timidity-executable-linux
		   timcfg-timidity-executable-win)
		 " -EB" bank
		 " -I" patch
		 " " timcfg-timidity-options " "
		 midi-test " &"))
      (message "no bank specified !"))
    (setq timcfg-last-bank bank
	  timcfg-last-patch patch)))
  

(defun timcfg-change-melody (&optional melody)
  "Invoke KeyKit for defining a new test melody."
  (interactive)
  (setq timcfg-melody (or melody (read-from-minibuffer "KeyKit phrase> ")))
  (with-key-or-lowkey "writemf(" timcfg-melody ", \"" timcfg-midi-test-file "\")")
  (if (timcfg-patch-line-p) (timcfg-listen-patch-at-point)))


;;; --------------------------- mixes ---------------------------

(defun timcfg-add-to-mix (&optional mix channel ph)
  (interactive)
  (let* ((bank (timcfg-current-bank))
	 (patch (timcfg-current-patch))
	 (mix (or mix (read-from-minibuffer "mix name> ")))
	 (ch (or channel (read-from-minibuffer "channel> " "1")))
	 (ph (read-from-minibuffer "phrase> " timcfg-melody)))
    (when (and bank patch)
      (end-of-line)
      (with-read-only-status
       (insert "\n#^" mix "^ " ch " " ph)))))


(defun timcfg-listen-mix-at-point ()
  "Call TiMidity and play the mix at point"
  (interactive)
  (let ((mix (timcfg-mix-line-p)))
    (when mix
      (with-key-or-lowkey
       (kk-eval 
	(concat "writemf(" (timcfg-get-mix-phrase mix) ",\"" timcfg-midi-mix-file "\")")))
      (shell-command
       (concat (if (equal system-type 'gnu/linux)
		   timcfg-timidity-executable-linux timcfg-timidity-executable-win)
	       " " timcfg-timidity-options " " timcfg-midi-mix-file " &")))))

(defun timcfg-get-mix-phrase (mix)
 (save-excursion
   (let ((mixph "''"))
     (goto-char (point-min))
     (while (re-search-forward (concat "#\\^" mix "\\^[ \t]\\([0-9]+\\)[ \t]\\(.*\\)$") nil t)
       (let* ((ch (match-string 1))
	      (ph (match-string 2))
	      (patch (save-excursion
		       (while (null (timcfg-patch-line-p))
			 (forward-line -1))
		       (concat "PourAC(" (timcfg-current-bank) "," 
			       (timcfg-current-patch) "," ch ")"))))
	 (setq mixph (concat mixph "|(" patch "+ SetChan(" ph "," ch "))"))))
     mixph)))

(defun timcfg-kill-line ()
  (with-read-only-status
   (beginning-of-line)
   (kill-line)
   (backward-char 1) 
   (delete-char 1)))

(defun timcfg-mix-move-up (n)
  (when (timcfg-mix-line-p)
    (timcfg-kill-line)
    (while (and (> n 0)
		(> (count-lines (point-min) (point)) 0))
      (if (timcfg-patch-line-p)
	  (setq n (- n 1)))
      (forward-line -1))
    (end-of-line)
    (with-read-only-status
     (insert "\n")
     (yank))))

(defun timcfg-mix-move-down (n)  
  (when (timcfg-mix-line-p)
    (timcfg-kill-line)
    (while (and (> n 0)
		(> (count-lines (point) (point-max)) 0))
      (if (timcfg-patch-line-p)
	  (setq n (- n 1)))
      (forward-line 1))
    (end-of-line)
    (with-read-only-status
     (insert "\n")
     (yank))))


;;; --------------------------- GeoMaestro interface -----------------------------


(defun with-key-or-lowkey (&rest command)
  "evaluate KeyKit COMMAND, preferably with KeyKit through TCP/IP,
else with lowkey if no connection is available"
  (if (and kk-tcpip-process (zerop (process-exit-status kk-tcpip-process)))
      (kk-eval (mapconcat 'identity command ""))
    (with-temp-buffer
      (insert (mapconcat 'identity command ""))
      (kk-run-lowkey "1" 1 (point-max)))))

(defmacro with-keykit-tcpip (&rest body)
  "ensure that a TCP-IP connection with KeyKit is enabled before evaluating BODY"
  `(when (or (and kk-tcpip-process (zerop (process-exit-status kk-tcpip-process)))
	       (if (yes-or-no-p  "We are not connected right now ! Start a connection ?")
		   (progn (kk-tcpip) t) nil))
	 ,@body))

(defmacro with-open-compositor (&rest body)
  "ensure that the LastGMCOMPO tool is open in current KeyKit page"
  `(if (string= "0" (kk-eval-with-return "BringCompositorUp()"))
       (message "Compositor was not found !")
     ,@body))

(defmacro with-open-gui (&rest body)
  "ensure that the LastGMGUI tool is open in current KeyKit page"
  `(if (string= "0" (kk-eval-with-return "BringGUIUp()"))
       (message "Main GUI was not found !")
     ,@body))


(defun timcfg-import-boxes (&optional channel) ;;;  amliorer...
  ""
  (interactive)
  (let* ((bank (timcfg-current-bank))
	 (patch (timcfg-current-patch))
	 (pos "xy(300,200)")
	 (dx "40")
	 (ch (or channel (read-from-minibuffer "channel: " "1")))
	 (ph (if (string= ch "1")
		 timcfg-melody
	       (concat "SetChan(" timcfg-melody "," ch ")"))))
    (if (and bank patch)
	(with-keykit-tcpip
	 (with-open-compositor
	  (kk-eval
	   (concat "LastGMCOMPOf.importation(string(" ph "), "
		   "[\"rx\"=" dx ",\"ry\"=0,\"nb\"="
		   (kk-eval-with-return 
		    (concat
		     "LastGMCOMPOf.importation(\"PourAC(" bank "," patch ","
		     ch ")\", " pos ", \"" bank "-" patch ":" ch "\")[\"i\"]"))
		   "], string(" ph "))")))))))


(defun timcfg-replace-box (&optional nb channel) 
  ""
  (interactive)
  (let* ((bank (timcfg-current-bank))
	 (patch (timcfg-current-patch))
	 (nb (or nb (read-from-minibuffer "box number: ")))
	 (ch (or channel (read-from-minibuffer "channel: " "1"))))
    (if (and bank patch)
	(with-keykit-tcpip
	 (with-open-compositor
	  (kk-eval
	   (concat
	    "LastGMCOMPOf.importation(\"PourAC(" bank "," patch "," ch ")\","
	    nb ", \"" bank "-" patch ":" ch "\")")))))))


(defun timcfg-snarf-patch (melodyp &optional channel var)
  ""
  (interactive)
  (let* ((bank (timcfg-current-bank))
	 (patch (timcfg-current-patch))
	 (ch (or channel (read-from-minibuffer "channel: " "1")))
	 (ph (if (string= ch "1")
		 timcfg-melody
	       (concat "SetChan(" timcfg-melody "," ch ")")))
	 (var (or var (read-from-minibuffer "variable: " "Snarf"))))
    (if (and bank patch)
	(with-keykit-tcpip
	 (kk-eval (concat var " = PourAC(" bank "," patch "," ch ")"
			  (if melodyp (concat "+" ph) "")))))))

(defun timcfg-snarf-mix ()
  (interactive)
  (let ((mix (timcfg-mix-line-p))
	(var (read-from-minibuffer "variable: " "Snarf")))
    (when mix
      (kk-eval (concat var "=" (timcfg-get-mix-phrase mix))))))


;;; --------------------------- Font Lock Mode  -----------------------------

(defconst timcfg-font-lock-keywords
  (list
   '("^\\(bank\\|dir\\|source\\|drumset\\).*$"
     0 font-lock-comment-face t)
   '("#.*$"
     0 font-lock-type-face t)
   '("^#\\^\\(.+\\)^" 
     1 font-lock-warning-face t)
   '("^#\\^.+^[ \t]\\([0-9]+\\)" 
     1 font-lock-keyword-face t)
   '("^#\\^.+^[ \t][0-9]+[ \t]\\(.*\\)$" 
     1 font-lock-comment-face t)
  "timidity-cfg-mode fontification"))

(defun timcfg-fontify ()
  "Loads the timcfg-font-lock-keywords into a local version of font-lock-keywords."
  (set (make-local-variable 'font-lock-defaults)
       '(timcfg-font-lock-keywords
         t t nil nil))
  (font-lock-mode 1))


;; this is it
(provide 'timidity-cfg)